#!/usr/bin/env python
"""
Cognitive Map Evaluation

This script evaluates cognitive map generation from various models.
It can be used in single-file mode or batch mode.

Usage:
    python eval_cogmap_main.py --input path/to/file.jsonl --output path/to/results.json
    python eval_cogmap_main.py --eval_dir path/to/dir --output_dir path/to/results
"""

import argparse
from utils.evaluation import evaluate_answers_and_cogmaps, batch_evaluate
from utils.io_utils import print_results, save_json_results

def main():
    """
    Main entry point for the cognitive map evaluation script.
    """
    parser = argparse.ArgumentParser(description='Evaluate model answers and generated cognitive maps')
    parser.add_argument('--input', type=str, default=None,
                        help='Path to a single JSONL file with model answers')
    parser.add_argument('--eval_dir', type=str, default=None,
                        help='Directory containing JSONL files for batch evaluation')
    parser.add_argument('--output', type=str, default=None,
                        help='Path to save the evaluation results (JSON format)')
    parser.add_argument('--output_dir', type=str, 
                        default=None,
                        help='Directory to save batch evaluation results')
    # output_dir can be used to organize evaluation results
    args = parser.parse_args()
    
    # Handle batch evaluation
    if args.eval_dir:
        print(f"Running batch evaluation on all JSONL files in {args.eval_dir}")
        batch_evaluate(args.eval_dir, args.output_dir)
        return
    
    # Single file evaluation
    input_path = args.input
    if not input_path:
        print("No input file specified. Use --input to specify a file to evaluate.")
        return
    
    # Evaluate answers and cognitive maps
    results, error_cases = evaluate_answers_and_cogmaps(input_path)
    
    # Print results to console
    print_results(results)
    
    # Save to file if output path is provided
    if args.output:
        save_json_results({
            'results': results,
            'error_cases': error_cases
        }, args.output)

if __name__ == "__main__":
    main()